%{
Copyright (c) 2011, MIT.
Developed with the sponsorship of the Defense Advanced Research Projects
Agency (DARPA).

Permission is hereby granted, free of charge, to any person obtaining a copy
of this data, including any software or models in source or binary form, as
well as any drawings, specifications, and documentation (collectively "the
Data"), to deal in the Data without restriction, including without
limitation the rights to use, copy, modify, merge, publish, distribute,
sublicense, and/or sell copies of the Data, and to permit persons to whom
the Data is furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE LIABLE
FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE DATA OR
THE USE OR OTHER DEALINGS IN THE DATA.
%}

% Matt Fitzgerald
% function to collapse given transition rule matrices into a single matrix

function [allRulePaths allDesignPaths allCostPaths] = ruleCollapse2(Tcost,Ttime,minArc,maxArc,fuzzy,hotStart)
% INPUTS
% Tcost {r}(td,td) - transition dollar costs
%                       0 = no transition
% Ttime {r}(td,td) - transition times for each rule
% minArc (scalar) - minimum # arc transitions considered
% maxArc (scalar) - maximum # arc transitions considered
% fuzzy (scalar) - fuzzy pareto % allowed
% hotStart - if it is 0 --> start at beginning
%            otherwise, it should be a row vector [rulePath] to be the
%            first rulePath to start on
% OUTPUTS
% allPaths {td,td} - cell array of non-dominated paths from here to there


% initialize variables
td = size(Tcost{1},1);
r = length(Tcost);
allRulePaths = cell(td,td);
allDesignPaths = cell(td,td);
allCostPaths = cell(td,td);
for i = 1:td
    for j = 1:td
        % these two need to be cell arrays of cells, since the entries will
        % be of different size
        allRulePaths{i,j} = {};
        allDesignPaths{i,j} = {};
    end
end

% construct permutations
permMat = cell(1,maxArc);
% one arc
permMat{1} = (1:r)';
% loop through remainder
for arcs = 2:maxArc
    prevSize = size(permMat{arcs-1},1);
    for rule = 1:r
        temp = [rule*ones(prevSize,1) permMat{arcs-1}];
        permMat{arcs} = [permMat{arcs} ; temp];
    end
end

% start a timer for saving purposes
tic;
backTime = 0;

% check hotStart
if all(hotStart)
    
    % find starting option number if hotStart is not zero
    optStart = 1;
    for i = 1:length(hotStart)
        optStart = optStart + r^(length(hotStart)-i) * (hotStart(i) - 1);
    end
    
    % abbreviated loop for the hotStart-length arcs
    for option = optStart:size(permMat{length(hotStart)},1)
        % set rule path
        rulePath = permMat{length(hotStart)}(option,:);
        
        % loop through all potential start designs
        for startDesign = 1:td
            [rulePath startDesign]
            
            % update allPaths for this rule path and this start design
            [allRulePaths allDesignPaths allCostPaths] = designPathTool(startDesign,rulePath,rulePath,Tcost,Ttime,allRulePaths,allDesignPaths,allCostPaths,fuzzy);
        end
        
        % check timer and save progress if it has been 20 minutes since
        % last rule path completion
        time = toc;
        if time > 60*20
            backTime = 0;
            save('RuleCollapseAutosave.mat','allRulePaths','allDesignPaths','allCostPaths');
        else
            backTime = backTime + time;
        end
        tic;
        
    end
    % main loop for remainder of duration
    % loop through permMat
    for arcs = (length(hotStart)+1):maxArc
        % loop through options in there
        for option = 1:size(permMat{arcs},1)
            % set rule path
            rulePath = permMat{arcs}(option,:);
            
            % loop through all potential start designs
            for startDesign = 1:td
                [rulePath startDesign]
                
                % update allPaths for this rule path and this start design
                [allRulePaths allDesignPaths allCostPaths] = designPathTool(startDesign,rulePath,rulePath,Tcost,Ttime,allRulePaths,allDesignPaths,allCostPaths,fuzzy);
            end
            
            % check timer and save progress if it has been 20 minutes since
            % last rule path completion
            time = toc;
            if time > 60*20
                backTime = 0;
                save('RuleCollapseAutosave.mat','allRulePaths','allDesignPaths','allCostPaths');
            else
                backTime = backTime + time;
            end
            tic;
        end
    end
    
    
    
    
else % if hotStart was 0, just start at the beginning
    % loop through permMat
    for arcs = minArc:maxArc
        % loop through options in there
        for option = 1:size(permMat{arcs},1)
            % set rule path
            rulePath = permMat{arcs}(option,:);
            
            % loop through all potential start designs
            for startDesign = 1:td
                [rulePath startDesign]
                
                % update allPaths for this rule path and this start design
                [allRulePaths allDesignPaths allCostPaths] = designPathTool(startDesign,rulePath,rulePath,Tcost,Ttime,allRulePaths,allDesignPaths,allCostPaths,fuzzy);
            end
            
            % check timer and save progress if it has been 20 minutes since
            % last rule path completion
            time = toc;
            if time > 60*20
                backTime = 0;
                save('RuleCollapseAutosave.mat','allRulePaths','allDesignPaths','allCostPaths');
            else
                backTime = backTime + time;
            end
            tic;
        end
    end
end

end